#!/bin/bash

# --- 配置 ---
CONTAINER_NAME="gitea"                      # Gitea 容器名称
BACKUP_DIR="/root/gitea"                    # 宿主机上用于存放备份的目录
TIMESTAMP=$(date +%Y%m%d_%H%M%S)
BACKUP_FILENAME="gitea_backup_${TIMESTAMP}.zip"
GITEA_CONFIG_PATH="/data/gitea/conf/app.ini" # 容器内的 Gitea 配置文件路径

# --- 确保备份目录存在 ---
mkdir -p "$BACKUP_DIR"

# --- 构建在容器内执行的备份命令 ---
DUMP_COMMAND="docker exec -u git -w /tmp ${CONTAINER_NAME} bash -c '/usr/local/bin/gitea dump -c ${GITEA_CONFIG_PATH}'"

echo "开始备份 Gitea 容器 '${CONTAINER_NAME}'..."
echo "在容器内执行命令: ${DUMP_COMMAND}"

# --- 执行备份命令 ---
if eval "$DUMP_COMMAND"; then
  echo "Gitea 容器备份完成，备份文件在容器的 /tmp 目录下。"

  # 获取实际的备份文件名
  CONTAINER_BACKUP_FILE=$(docker exec ${CONTAINER_NAME} sh -c "ls /tmp/gitea-dump-*.zip 2>/dev/null | head -n 1")

  if [ -z "$CONTAINER_BACKUP_FILE" ]; then
    echo "未找到容器内的备份文件，备份失败！"
    exit 1
  fi

  HOST_BACKUP_PATH="${BACKUP_DIR}/${BACKUP_FILENAME}"

  echo "从容器复制备份文件 '${CONTAINER_BACKUP_FILE}' 到宿主机 '${HOST_BACKUP_PATH}'..."
  docker cp "${CONTAINER_NAME}:${CONTAINER_BACKUP_FILE}" "${HOST_BACKUP_PATH}"

  if [ $? -eq 0 ]; then
    echo "备份文件成功复制到宿主机: ${HOST_BACKUP_PATH}"

    # --- （可选）清理容器内的临时备份文件 ---
    docker exec "${CONTAINER_NAME}" bash -c "rm -f /tmp/gitea-dump-*.zip"
    echo "已清理容器内的临时备份文件。"
  else
    echo "警告：复制备份文件到宿主机失败！"
  fi
else
  echo "错误：Gitea 容器备份失败！请检查容器是否正在运行以及 Gitea 的配置。"
  exit 1
fi

echo "自动备份脚本执行完毕。"

# --- （可选）清理旧备份文件 ---
cd "$BACKUP_DIR"
ls -1t gitea_backup_*.zip | tail -n +2 | xargs -r rm -f
exit 0